<?php

namespace GiftCity\WooCommerce;

use WP_Error;

if (!defined('ABSPATH')) {
    exit;
}

class Client
{
    private static ?Client $instance = null;

    private string $baseUrl;
    private string $apiKey;

    public static function instance(): Client
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct()
    {
        $settings = giftcity_wc_get_settings();
        $this->baseUrl = rtrim($settings['api_base'] ?? 'https://gift30t.com/api/v1', '/');
        $this->apiKey = trim((string) ($settings['api_key'] ?? ''));
    }

    public function refreshSettings(): void
    {
        $settings = giftcity_wc_get_settings();
        $this->baseUrl = rtrim($settings['api_base'] ?? 'https://gift30t.com/api/v1', '/');
        $this->apiKey = trim((string) ($settings['api_key'] ?? ''));
    }

    /**
     * @throws \RuntimeException
     */
    public function request(string $method, string $path, array $args = []): array
    {
        if (empty($this->apiKey)) {
            throw new \RuntimeException(__('لطفاً کلید API گیفت سیتی را در تنظیمات افزونه ثبت کنید.', 'giftcity-wc'));
        }

        $method = strtoupper($method);
        $url = $this->buildUrl($path, $args['query'] ?? []);

        $requestArgs = [
            'method'  => $method,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept'       => 'application/json',
                'x-api-key'    => $this->apiKey,
            ],
            'timeout' => 25,
        ];

        if (isset($args['body'])) {
            $requestArgs['body'] = wp_json_encode($args['body']);
        }

        $response = wp_remote_request($url, $requestArgs);

        if (is_wp_error($response)) {
            throw new \RuntimeException($response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);

        if ($code >= 400) {
            $message = $decoded['message'] ?? sprintf(__('خطای نامشخص (%d)', 'giftcity-wc'), $code);
            throw new \RuntimeException($message);
        }

        if (!is_array($decoded)) {
            throw new \RuntimeException(__('پاسخ نامعتبر از سرور دریافت شد.', 'giftcity-wc'));
        }

        return $decoded;
    }

    private function buildUrl(string $path, array $query = []): string
    {
        $path = '/' . ltrim($path, '/');
        $url = $this->baseUrl . $path;

        if (!empty($query)) {
            $url = add_query_arg($query, $url);
        }

        return $url;
    }

    public function getProducts(int $page = 1, int $pageSize = 50, array $filters = []): array
    {
        $query = array_merge([
            'page'     => $page,
            'pageSize' => $pageSize,
        ], $filters);

        $response = $this->request('GET', '/products', ['query' => $query]);
        return [
            'items' => $response['data'] ?? [],
            'meta'  => $response['meta']['pagination'] ?? [],
        ];
    }

    public function getProductDetails(int $productId): array
    {
        $response = $this->request('GET', '/products/' . $productId);
        return $response['data'] ?? [];
    }

    public function getCategories(): array
    {
        $response = $this->request('GET', '/products/categories');
        return $response['data'] ?? [];
    }

    public function getWalletBalance(): array
    {
        $response = $this->request('GET', '/wallet/balance');
        return $response['data'] ?? [];
    }

    public function createOrder(array $payload): array
    {
        $response = $this->request('POST', '/orders', ['body' => $payload]);
        return $response['data'] ?? [];
    }
}
