<?php

namespace GiftCity\WooCommerce;

if (!defined('ABSPATH')) {
    exit;
}

class Order_Service
{
    private Client $client;

    public function __construct()
    {
        $this->client = Client::instance();

        add_action('woocommerce_order_status_processing', [$this, 'maybeSendOrder'], 10, 1);
        add_action('woocommerce_order_status_completed', [$this, 'maybeSendOrder'], 10, 1);
        add_filter('woocommerce_order_actions', [$this, 'registerOrderAction']);
        add_action('woocommerce_order_action_giftcity_wc_resend_order', [$this, 'manualOrderSend']);
    }

    public function registerOrderAction(array $actions): array
    {
        global $theorder;
        if ($theorder && $this->containsGiftCityProducts($theorder)) {
            $actions['giftcity_wc_resend_order'] = __('ارسال مجدد به گیفت سیتی', 'giftcity-wc');
        }
        return $actions;
    }

    public function manualOrderSend($order): void
    {
        if ($order instanceof \WC_Order) {
            $this->maybeSendOrder($order->get_id(), true);
        }
    }

    public function maybeSendOrder($orderId, bool $force = false): void
    {
        $order = wc_get_order($orderId);
        if (!$order instanceof \WC_Order) {
            return;
        }

        if (!$force && 'yes' === $order->get_meta('_giftcity_remote_synced')) {
            return;
        }

        $payload = $this->buildPayload($order);
        if (empty($payload['products'])) {
            return;
        }

        try {
            $response = $this->client->createOrder($payload);
            $reference = $response['orderReference'] ?? ($response['orderId'] ?? '');
            $order->update_meta_data('_giftcity_remote_synced', 'yes');
            if ($reference) {
                $order->update_meta_data('_giftcity_remote_reference', $reference);
            }
            $order->add_order_note(__('سفارش با موفقیت به گیفت سیتی ارسال شد.', 'giftcity-wc'));
            if ($reference) {
                $order->add_order_note(sprintf(__('کد پیگیری گیفت سیتی: %s', 'giftcity-wc'), $reference));
            }
            $order->save();
        } catch (\Throwable $e) {
            $order->add_order_note(sprintf(__('خطا در ارتباط با گیفت سیتی: %s', 'giftcity-wc'), $e->getMessage()));
            giftcity_wc_log('Order push failed #' . $order->get_id() . ': ' . $e->getMessage(), 'error');
        }
    }

    private function buildPayload(\WC_Order $order): array
    {
        $products = [];
        $requiredFields = [];

        foreach ($order->get_items('line_item') as $itemId => $item) {
            $wcProduct = $item->get_product();
            if (!$wcProduct) {
                continue;
            }

            $productId = intval(get_post_meta($wcProduct->get_id(), '_giftcity_product_id', true));
            $propertyId = intval(get_post_meta($wcProduct->get_id(), '_giftcity_region_property_id', true));

            if ($productId <= 0 || $propertyId <= 0) {
                continue;
            }

            $products[] = [
                'productId'       => $productId,
                'regionPropertyId'=> $propertyId,
                'quantity'        => max(1, (int) $item->get_quantity()),
            ];

            $fields = $this->extractRequiredFields($item);
            if (!empty($fields)) {
                $requiredFields[$productId] = $fields;
            }
        }

        $payload = ['products' => $products];
        if (!empty($requiredFields)) {
            $payload['requiredFields'] = $requiredFields;
        }

        return $payload;
    }

    private function extractRequiredFields($item): array
    {
        $fields = [];
        foreach ($item->get_meta_data() as $meta) {
            if (strpos($meta->key, '_giftcity_field_') !== 0) {
                continue;
            }

            $value = json_decode($meta->value, true);
            if (isset($value['id'], $value['value'])) {
                $fields[] = [
                    'fieldId' => intval($value['id']),
                    'value'   => sanitize_text_field($value['value']),
                ];
            }
        }
        return $fields;
    }

    private function containsGiftCityProducts(\WC_Order $order): bool
    {
        foreach ($order->get_items('line_item') as $item) {
            $wcProduct = $item->get_product();
            if ($wcProduct && get_post_meta($wcProduct->get_id(), '_giftcity_product_id', true)) {
                return true;
            }
        }
        return false;
    }
}
