<?php

namespace GiftCity\WooCommerce;

use WC_Product_Simple;

if (!defined('ABSPATH')) {
    exit;
}

class Sync_Service
{
    private Client $client;

    public function __construct()
    {
        $this->client = Client::instance();

        add_filter('cron_schedules', [__CLASS__, 'registerIntervals']);
        add_action('giftcity_wc_sync_products_event', [$this, 'syncProducts']);
        add_action('admin_post_giftcity_wc_manual_sync', [$this, 'handleManualSync']);
        add_action('admin_post_giftcity_wc_import_single', [$this, 'handleSingleImport']);
        add_action('admin_notices', [$this, 'renderNotices']);
        add_action('woocommerce_product_options_pricing', [$this, 'renderMarkupField']);
        add_action('woocommerce_admin_process_product_object', [$this, 'saveMarkupField']);
    }

    public static function registerIntervals($schedules)
    {
        $schedules['giftcity_fifteen_minutes'] = [
            'interval' => 15 * 60,
            'display'  => __('هر ۱۵ دقیقه', 'giftcity-wc'),
        ];
        return $schedules;
    }

    private static function ensureIntervalFilter(): void
    {
        if (!has_filter('cron_schedules', [__CLASS__, 'registerIntervals'])) {
            add_filter('cron_schedules', [__CLASS__, 'registerIntervals']);
        }
    }

    public static function schedule_event(): void
    {
        self::ensureIntervalFilter();
        $interval = giftcity_wc_get_option('sync_interval', 'hourly');
        if (!wp_next_scheduled('giftcity_wc_sync_products_event')) {
            wp_schedule_event(time() + 60, $interval, 'giftcity_wc_sync_products_event');
        }
    }

    public static function unschedule_event(): void
    {
        $timestamp = wp_next_scheduled('giftcity_wc_sync_products_event');
        while ($timestamp) {
            wp_unschedule_event($timestamp, 'giftcity_wc_sync_products_event');
            $timestamp = wp_next_scheduled('giftcity_wc_sync_products_event');
        }
    }

    public static function reschedule_event(string $interval): void
    {
        self::ensureIntervalFilter();
        self::unschedule_event();
        wp_schedule_event(time() + 60, $interval, 'giftcity_wc_sync_products_event');
    }

    public function handleManualSync(): void
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('دسترسی غیرمجاز', 'giftcity-wc'));
        }

        check_admin_referer('giftcity_wc_manual_sync');
        $count = $this->syncProducts(true);
        $this->addNotice(sprintf(__('سنک محصولات با موفقیت انجام شد (%d قلم).', 'giftcity-wc'), $count));
        wp_safe_redirect(admin_url('admin.php?page=giftcity-wc-settings'));
        exit;
    }

    public function handleSingleImport(): void
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('دسترسی غیرمجاز', 'giftcity-wc'));
        }

        check_admin_referer('giftcity_wc_import_single');

        $productId = isset($_POST['giftcity_product_id']) ? intval($_POST['giftcity_product_id']) : 0;
        $propertyId = isset($_POST['giftcity_property_id']) ? intval($_POST['giftcity_property_id']) : null;
        $wooCategoryId = isset($_POST['giftcity_wc_term_id']) ? intval($_POST['giftcity_wc_term_id']) : 0;

        if ($productId <= 0) {
            $this->addNotice(__('شناسه محصول معتبر نیست.', 'giftcity-wc'), 'error');
            wp_safe_redirect(admin_url('admin.php?page=giftcity-wc-settings'));
            exit;
        }

        if ($wooCategoryId <= 0 || !term_exists($wooCategoryId, 'product_cat')) {
            $this->addNotice(__('لطفاً یک دسته‌بندی معتبر از ووکامرس انتخاب کنید.', 'giftcity-wc'), 'error');
            wp_safe_redirect(admin_url('admin.php?page=giftcity-wc-settings'));
            exit;
        }

        try {
            $product = $this->client->getProductDetails($productId);
            $count = $this->importProduct($product, $propertyId, $wooCategoryId);
            if ($count > 0) {
                $this->addNotice(sprintf(__('محصول/پکیج انتخابی با موفقیت اضافه شد (%d مورد).', 'giftcity-wc'), $count));
            } else {
                $this->addNotice(__('پراپرتی موردنظر یافت نشد.', 'giftcity-wc'), 'error');
            }
        } catch (\Throwable $e) {
            $this->addNotice($e->getMessage(), 'error');
        }

        wp_safe_redirect(admin_url('admin.php?page=giftcity-wc-settings'));
        exit;
    }

    public function renderNotices(): void
    {
        $notice = get_transient('giftcity_wc_notice');
        if (!$notice) {
            return;
        }

        delete_transient('giftcity_wc_notice');
        $class = 'notice notice-' . esc_attr($notice['type'] ?? 'success');
        printf('<div class="%1$s is-dismissible"><p>%2$s</p></div>', $class, esc_html($notice['message']));
    }

    private function addNotice(string $message, string $type = 'success'): void
    {
        set_transient('giftcity_wc_notice', [
            'message' => $message,
            'type'    => $type === 'error' ? 'error' : 'success',
        ], 60);
    }

    /**
     * همگام‌سازی تمام محصولات قابل دسترس در API.
     */
    public function syncProducts(bool $manual = false): int
    {
        if (!giftcity_wc_is_woocommerce_active()) {
            return 0;
        }

        @set_time_limit(0);
        $totalImported = 0;
        $page = 1;
        $totalPages = 1;

        do {
            try {
                $response = $this->client->getProducts($page, 50, []);
            } catch (\Throwable $e) {
                giftcity_wc_log('Product list sync failed: ' . $e->getMessage(), 'error');
                break;
            }

            $items = $response['items'] ?? [];
            foreach ($items as $item) {
                try {
                    $details = $this->client->getProductDetails((int) $item['id']);
                    $totalImported += $this->importProduct($details);
                } catch (\Throwable $e) {
                    giftcity_wc_log('Failed to import product #' . ($item['id'] ?? 'unknown') . ': ' . $e->getMessage(), 'error');
                }
            }

            $pagination = $response['meta'] ?? [];
            $totalPages = max(1, intval($pagination['totalPages'] ?? $totalPages));
            $page++;
        } while ($page <= $totalPages);

        return $totalImported;
    }

    private function importProduct(array $product, ?int $propertyFilter = null, ?int $forcedCategoryId = null): int
    {
        if (empty($product['regions'])) {
            return 0;
        }

        $count = 0;
        foreach ($product['regions'] as $region) {
            if (empty($region['properties'])) {
                continue;
            }

            foreach ($region['properties'] as $property) {
                if ($propertyFilter && intval($property['id']) !== $propertyFilter) {
                    continue;
                }

                $result = $this->upsertWooProduct($product, $region, $property, $forcedCategoryId);
                if ($result) {
                    $count++;
                }
            }
        }

        return $count;
    }

    private function upsertWooProduct(array $product, array $region, array $property, ?int $forcedCategoryId = null): int
    {
        $sku = sprintf('giftcity-%d-%d', $product['id'], $property['id']);
        $postId = wc_get_product_id_by_sku($sku);
        $isNew = false;

        $title = sprintf('%s - %s (%s)', $product['name'] ?? __('محصول گیفت سیتی', 'giftcity-wc'), $region['name'] ?? '', $property['title'] ?? '');

        if (!$postId) {
            $postId = wp_insert_post([
                'post_title'     => $title,
                'post_type'      => 'product',
                'post_status'    => 'publish',
                'post_content'   => $product['description'] ?? '',
                'post_excerpt'   => wp_trim_words(wp_strip_all_tags($product['description'] ?? ''), 40, '...'),
                'comment_status' => 'closed',
            ]);

            if (is_wp_error($postId) || !$postId) {
                giftcity_wc_log('Unable to insert product: ' . $title, 'error');
                return 0;
            }

            $isNew = true;
        } else {
            wp_update_post([
                'ID'           => $postId,
                'post_title'   => $title,
                'post_content' => $product['description'] ?? '',
            ]);
        }

        $wcProduct = wc_get_product($postId);
        if (!$wcProduct) {
            $wcProduct = new WC_Product_Simple($postId);
        }

        $basePrice = floatval($property['price'] ?? 0);
        $finalPrice = $this->calculatePrice($basePrice, $postId);

        $wcProduct->set_sku($sku);
        $wcProduct->set_regular_price((string) $finalPrice);
        $wcProduct->set_price((string) $finalPrice);
        $wcProduct->set_virtual(true);
        $wcProduct->set_manage_stock(false);
        $wcProduct->set_catalog_visibility('visible');
        $wcProduct->set_status('publish');
        $wcProduct->set_sold_individually(false);
        $wcProduct->set_stock_status(($property['stock_status'] ?? '') === 'OUTSTOCK' ? 'outofstock' : 'instock');
        $wcProduct->save();

        update_post_meta($postId, '_giftcity_product_id', intval($product['id']));
        update_post_meta($postId, '_giftcity_region_id', intval($region['id']));
        update_post_meta($postId, '_giftcity_region_property_id', intval($property['id']));
        update_post_meta($postId, '_giftcity_region_name', sanitize_text_field($region['name'] ?? ''));
        update_post_meta($postId, '_giftcity_property_title', sanitize_text_field($property['title'] ?? ''));
        update_post_meta($postId, '_giftcity_base_price', $basePrice);
        update_post_meta($postId, '_giftcity_last_synced', current_time('mysql'));

        if (!empty($product['requiredFields'])) {
            update_post_meta($postId, '_giftcity_required_fields', wp_json_encode($product['requiredFields']));
        } else {
            delete_post_meta($postId, '_giftcity_required_fields');
        }

        $this->assignCategories($postId, $forcedCategoryId);

        return $postId;
    }

    private function calculatePrice(float $basePrice, int $postId): float
    {
        $markup = get_post_meta($postId, '_giftcity_markup_percent', true);
        if ($markup === '' || $markup === false) {
            $markup = giftcity_wc_get_option('default_markup', 0);
        }

        $markup = floatval($markup);
        $price = $basePrice * (1 + ($markup / 100));
        return max(1, round($price));
    }

    private function ensureCategory(): int
    {
        $term = get_term_by('slug', 'giftcity-products', 'product_cat');
        if ($term && !is_wp_error($term)) {
            return intval($term->term_id);
        }

        $result = wp_insert_term(__('محصولات گیفت سیتی', 'giftcity-wc'), 'product_cat', [
            'slug' => 'giftcity-products',
        ]);

        if (is_wp_error($result)) {
            return 0;
        }

        return intval($result['term_id']);
    }

    private function assignCategories(int $postId, ?int $selectedCategoryId): void
    {
        $categories = [];

        if ($selectedCategoryId && term_exists($selectedCategoryId, 'product_cat')) {
            $categories[] = $selectedCategoryId;
        }

        $defaultCategory = $this->ensureCategory();
        if ($defaultCategory && (!$selectedCategoryId || $defaultCategory !== $selectedCategoryId)) {
            $categories[] = $defaultCategory;
        }

        if (!empty($categories)) {
            $categories = array_unique(array_map('intval', $categories));
            wp_set_object_terms($postId, $categories, 'product_cat', true);
        }
    }

    public function renderMarkupField(): void
    {
        global $post;
        if (!$post) {
            return;
        }

        $productId = $post->ID;
        if (!get_post_meta($productId, '_giftcity_product_id', true)) {
            return;
        }

        $value = get_post_meta($productId, '_giftcity_markup_percent', true);
        echo '<div class="options_group">';
        woocommerce_wp_text_input([
            'id'          => '_giftcity_markup_percent',
            'label'       => __('درصد سود اختصاصی گیفت سیتی', 'giftcity-wc'),
            'desc_tip'    => true,
            'description' => __('در صورت خالی بودن، مقدار پیش‌فرض تنظیمات استفاده می‌شود.', 'giftcity-wc'),
            'type'        => 'number',
            'attributes'  => [
                'step' => '0.1',
                'min'  => '0',
            ],
            'value'       => $value,
        ]);
        echo '</div>';
    }

    public function saveMarkupField($product): void
    {
        if (!$product instanceof \WC_Product) {
            return;
        }

        if (!get_post_meta($product->get_id(), '_giftcity_product_id', true)) {
            return;
        }

        $value = isset($_POST['_giftcity_markup_percent']) ? sanitize_text_field($_POST['_giftcity_markup_percent']) : '';
        if ($value === '') {
            delete_post_meta($product->get_id(), '_giftcity_markup_percent');
        } else {
            update_post_meta($product->get_id(), '_giftcity_markup_percent', floatval($value));
        }

        $basePrice = floatval(get_post_meta($product->get_id(), '_giftcity_base_price', true));
        if ($basePrice > 0) {
            $newPrice = $this->calculatePrice($basePrice, $product->get_id());
            $product->set_regular_price((string) $newPrice);
            $product->set_price((string) $newPrice);
        }
    }
}
