<?php

namespace GiftCity\WooCommerce;

if (!defined('ABSPATH')) {
    exit;
}

class Plugin
{
    private static ?Plugin $instance = null;

    private ?Settings $settings = null;
    private ?Sync_Service $syncService = null;
    private ?Order_Service $orderService = null;

    public static function instance(): Plugin
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    private function __construct()
    {
        add_action('plugins_loaded', [$this, 'boot']);
        add_action('init', [$this, 'loadTextDomain']);
    }

    public static function activate(): void
    {
        if (!giftcity_wc_is_woocommerce_active()) {
            deactivate_plugins(plugin_basename(GIFTCITY_WC_PLUGIN_FILE));
            wp_die(__('برای استفاده از این افزونه باید ووکامرس فعال باشد.', 'giftcity-wc'));
        }

        Sync_Service::schedule_event();
    }

    public static function deactivate(): void
    {
        Sync_Service::unschedule_event();
    }

    public function loadTextDomain(): void
    {
        load_plugin_textdomain('giftcity-wc', false, dirname(plugin_basename(GIFTCITY_WC_PLUGIN_FILE)) . '/languages');
    }

    public function boot(): void
    {
        if (!giftcity_wc_is_woocommerce_active()) {
            add_action('admin_notices', [$this, 'woocommerceMissingNotice']);
            return;
        }

        Client::instance()->refreshSettings();

        $this->settings = new Settings();
        $this->syncService = new Sync_Service();
        $this->orderService = new Order_Service();

        $this->registerProductFieldHooks();
    }

    public function woocommerceMissingNotice(): void
    {
        echo '<div class="notice notice-error"><p>' . esc_html__('برای اجرای افزونهٔ گیفت سیتی باید ووکامرس نصب و فعال باشد.', 'giftcity-wc') . '</p></div>';
    }

    private function registerProductFieldHooks(): void
    {
        add_action('woocommerce_before_add_to_cart_button', [$this, 'renderProductFields']);
        add_filter('woocommerce_add_to_cart_validation', [$this, 'validateProductFields'], 10, 2);
        add_filter('woocommerce_add_cart_item_data', [$this, 'storeCartItemFields'], 10, 3);
        add_filter('woocommerce_get_item_data', [$this, 'displayCartItemFields'], 10, 2);
        add_action('woocommerce_checkout_create_order_line_item', [$this, 'attachFieldsToOrderItem'], 10, 4);
    }

    private function getRequiredFields(int $productId): array
    {
        $raw = get_post_meta($productId, '_giftcity_required_fields', true);
        if (!$raw) {
            return [];
        }

        $decoded = json_decode($raw, true);
        return is_array($decoded) ? $decoded : [];
    }

    public function renderProductFields(): void
    {
        global $product;

        if (!$product instanceof \WC_Product) {
            return;
        }

        $fields = $this->getRequiredFields($product->get_id());
        if (empty($fields)) {
            return;
        }

        wp_nonce_field('giftcity_wc_product_fields', 'giftcity_wc_product_fields_nonce');

        echo '<div class="giftcity-wc-fields">';
        echo '<h4>' . esc_html__('اطلاعات لازم برای ثبت سفارش', 'giftcity-wc') . '</h4>';

        foreach ($fields as $field) {
            $fieldId = intval($field['id']);
            $labelText = isset($field['fieldName']) && $field['fieldName'] !== ''
                ? wp_strip_all_tags($field['fieldName'])
                : sprintf(__('فیلد %d', 'giftcity-wc'), $fieldId);
            $inputName = 'giftcity_field_' . $fieldId;
            $type = $field['fieldType'] ?? 'text';
            $label = esc_html($labelText);

            echo '<p class="form-row form-row-wide">';
            echo '<label for="' . esc_attr($inputName) . '">' . $label . ' <span class="required">*</span></label>';
            printf(
                '<input type="%1$s" name="%2$s" id="%2$s" value="%3$s" required class="input-text" />',
                esc_attr($this->mapFieldType($type)),
                esc_attr($inputName),
                isset($_POST[$inputName]) ? esc_attr(wp_unslash($_POST[$inputName])) : ''
            );
            echo '</p>';
        }

        echo '</div>';
    }

    private function mapFieldType(string $type): string
    {
        $type = strtolower($type);
        if (in_array($type, ['email', 'tel', 'number'], true)) {
            return $type;
        }
        return 'text';
    }

    public function validateProductFields($passed, $productId)
    {
        $fields = $this->getRequiredFields((int) $productId);
        if (empty($fields)) {
            return $passed;
        }

        if (!isset($_POST['giftcity_wc_product_fields_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['giftcity_wc_product_fields_nonce'])), 'giftcity_wc_product_fields')) {
            wc_add_notice(__('اعتبارسنجی ناموفق بود. لطفاً دوباره تلاش کنید.', 'giftcity-wc'), 'error');
            return false;
        }

        foreach ($fields as $field) {
            $fieldId = intval($field['id']);
            $inputName = 'giftcity_field_' . $fieldId;
            $value = isset($_POST[$inputName]) ? sanitize_text_field(wp_unslash($_POST[$inputName])) : '';

            if ('' === $value) {
                $labelText = isset($field['fieldName']) && $field['fieldName'] !== ''
                    ? wp_strip_all_tags($field['fieldName'])
                    : sprintf(__('فیلد %d', 'giftcity-wc'), $fieldId);
                wc_add_notice(sprintf(__('فیلد %s الزامی است.', 'giftcity-wc'), $labelText), 'error');
                return false;
            }
        }

        return $passed;
    }

    public function storeCartItemFields($cartItemData, $productId, $variationId)
    {
        $fields = $this->getRequiredFields((int) $productId);
        if (empty($fields)) {
            return $cartItemData;
        }

        $values = [];
        foreach ($fields as $field) {
            $fieldId = intval($field['id']);
            $inputName = 'giftcity_field_' . $fieldId;
            $value = isset($_POST[$inputName]) ? sanitize_text_field(wp_unslash($_POST[$inputName])) : '';
            if ($value !== '') {
                $labelText = isset($field['fieldName']) && $field['fieldName'] !== ''
                    ? wp_strip_all_tags($field['fieldName'])
                    : sprintf(__('فیلد %d', 'giftcity-wc'), $fieldId);
                $values[$fieldId] = [
                    'label' => $labelText,
                    'value' => $value,
                ];
            }
        }

        if (!empty($values)) {
            $cartItemData['giftcity_required_fields'] = $values;
        }

        return $cartItemData;
    }

    public function displayCartItemFields(array $itemData, array $cartItem): array
    {
        if (isset($cartItem['giftcity_required_fields'])) {
            foreach ($cartItem['giftcity_required_fields'] as $field) {
                $itemData[] = [
                    'name'  => sanitize_text_field($field['label']),
                    'value' => esc_html($field['value']),
                ];
            }
        }

        return $itemData;
    }

    public function attachFieldsToOrderItem($item, $cartItemKey, $values, $order): void
    {
        if (isset($values['giftcity_required_fields'])) {
            foreach ($values['giftcity_required_fields'] as $fieldId => $field) {
                $item->add_meta_data('_giftcity_field_' . $fieldId, wp_json_encode([
                    'id'    => $fieldId,
                    'label' => $field['label'],
                    'value' => $field['value'],
                ]), true);
            }
        }
    }
}
