<?php

namespace GiftCity\WooCommerce;

if (!defined('ABSPATH')) {
    exit;
}

class Settings
{
    private Client $client;

    public function __construct()
    {
        $this->client = Client::instance();

        add_action('admin_menu', [$this, 'registerMenu']);
        add_action('admin_init', [$this, 'registerSettings']);
    }

    public function registerMenu(): void
    {
        add_submenu_page(
            'woocommerce',
            __('افزونه گیفت سیتی', 'giftcity-wc'),
            __('گیفت سیتی', 'giftcity-wc'),
            'manage_woocommerce',
            'giftcity-wc-settings',
            [$this, 'renderPage'],
            60
        );
    }

    public function registerSettings(): void
    {
        register_setting('giftcity_wc_settings_group', 'giftcity_wc_settings', [
            'sanitize_callback' => [$this, 'sanitizeSettings'],
        ]);

        add_settings_section('giftcity_wc_general', __('اتصال به API', 'giftcity-wc'), function () {
            echo '<p>' . esc_html__('برای استفاده از وب‌سرویس، آدرس و کلید API خود را وارد کنید.', 'giftcity-wc') . '</p>';
        }, 'giftcity_wc_settings');

        add_settings_field('giftcity_wc_api_base', __('آدرس پایه API', 'giftcity-wc'), function () {
            $value = esc_url(giftcity_wc_get_option('api_base', 'https://gift30t.com/api/v1'));
            printf('<input type="url" name="giftcity_wc_settings[api_base]" value="%s" class="regular-text ltr" />', $value);
        }, 'giftcity_wc_settings', 'giftcity_wc_general');

        add_settings_field('giftcity_wc_api_key', __('کلید API', 'giftcity-wc'), function () {
            $value = esc_attr(giftcity_wc_get_option('api_key'));
            printf('<input type="password" name="giftcity_wc_settings[api_key]" value="%s" class="regular-text ltr" autocomplete="off" />', $value);
            echo '<p class="description">' . esc_html__('این کلید از پنل گیفت سیتی برای شما صادر می‌شود.', 'giftcity-wc') . '</p>';
        }, 'giftcity_wc_settings', 'giftcity_wc_general');

        add_settings_section('giftcity_wc_sync', __('تنظیمات همگام‌سازی و قیمت', 'giftcity-wc'), function () {
            echo '<p>' . esc_html__('بازه سنک خودکار و درصد سود پیش‌فرض را مشخص کنید.', 'giftcity-wc') . '</p>';
        }, 'giftcity_wc_settings');

        add_settings_field('giftcity_wc_default_markup', __('درصد سود پیش‌فرض', 'giftcity-wc'), function () {
            $value = floatval(giftcity_wc_get_option('default_markup', 10));
            printf('<input type="number" step="0.1" min="0" name="giftcity_wc_settings[default_markup]" value="%s" /> %%', esc_attr($value));
            echo '<p class="description">' . esc_html__('قیمت ووکامرس = قیمت گیفت سیتی × (۱ + درصد سود / ۱۰۰)', 'giftcity-wc') . '</p>';
        }, 'giftcity_wc_settings', 'giftcity_wc_sync');

        add_settings_field('giftcity_wc_sync_interval', __('بازه سنک خودکار', 'giftcity-wc'), function () {
            $value = giftcity_wc_get_option('sync_interval', 'hourly');
            $options = [
                'giftcity_fifteen_minutes' => __('هر ۱۵ دقیقه', 'giftcity-wc'),
                'hourly'                  => __('ساعتی', 'giftcity-wc'),
                'twicedaily'              => __('دو بار در روز', 'giftcity-wc'),
                'daily'                   => __('روزانه', 'giftcity-wc'),
            ];
            echo '<select name="giftcity_wc_settings[sync_interval]">';
            foreach ($options as $key => $label) {
                printf('<option value="%1$s" %2$s>%3$s</option>', esc_attr($key), selected($value, $key, false), esc_html($label));
            }
            echo '</select>';
        }, 'giftcity_wc_settings', 'giftcity_wc_sync');
    }

    public function sanitizeSettings(array $input): array
    {
        $sanitized = [];
        $sanitized['api_base'] = isset($input['api_base']) ? esc_url_raw($input['api_base']) : 'https://gift30t.com/api/v1';
        $sanitized['api_key'] = isset($input['api_key']) ? sanitize_text_field($input['api_key']) : '';
        $sanitized['default_markup'] = isset($input['default_markup']) ? floatval($input['default_markup']) : 0;
        $sanitized['sync_interval'] = isset($input['sync_interval']) ? sanitize_text_field($input['sync_interval']) : 'hourly';

        Client::instance()->refreshSettings();
        Sync_Service::reschedule_event($sanitized['sync_interval']);

        return $sanitized;
    }

    public function renderPage(): void
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('شما دسترسی لازم را ندارید.', 'giftcity-wc'));
        }

        $wallet = $this->maybeGetWalletBalance();
        $syncedCount = $this->getSyncedProductsCount();
        $remoteProducts = $this->getRemoteProducts();
        $wooCategories = $this->getWooProductCategories();
        ?>
        <div class="wrap giftcity-wc-admin">
            <h1><?php esc_html_e('افزونه فروش گیفت کارت گیفت سیتی', 'giftcity-wc'); ?></h1>

            <div class="giftcity-wc-panels">
                <div class="giftcity-panel">
                    <h2><?php esc_html_e('کیف پول وب‌سرویس', 'giftcity-wc'); ?></h2>
                    <?php if ($wallet): ?>
                        <p><strong><?php esc_html_e('موجودی فعلی:', 'giftcity-wc'); ?></strong> <?php echo esc_html(giftcity_wc_format_money($wallet['balance'] ?? 0)); ?></p>
                        <p class="description"><?php esc_html_e('اطمینان حاصل کنید موجودی همیشه از جمع سفارش‌های معلق بیشتر باشد.', 'giftcity-wc'); ?></p>
                    <?php else: ?>
                        <p class="description"><?php esc_html_e('برای مشاهده موجودی، کلید API معتبر وارد کنید.', 'giftcity-wc'); ?></p>
                    <?php endif; ?>
                </div>

                <div class="giftcity-panel">
                    <h2><?php esc_html_e('وضعیت سنک', 'giftcity-wc'); ?></h2>
                    <p><strong><?php esc_html_e('تعداد محصولات واردشده:', 'giftcity-wc'); ?></strong> <?php echo esc_html($syncedCount); ?></p>
                    <p><strong><?php esc_html_e('بازه خودکار:', 'giftcity-wc'); ?></strong> <?php echo esc_html($this->translateInterval(giftcity_wc_get_option('sync_interval', 'hourly'))); ?></p>
                </div>
            </div>

            <form method="post" action="options.php">
                <?php
                settings_fields('giftcity_wc_settings_group');
                do_settings_sections('giftcity_wc_settings');
                submit_button();
                ?>
            </form>

            <hr />

            <h2><?php esc_html_e('مدیریت سنک محصولات', 'giftcity-wc'); ?></h2>
            <p><?php esc_html_e('می‌توانید همه محصولات را یکجا سنک کنید یا فقط یک محصول/پکیج خاص را اضافه کنید.', 'giftcity-wc'); ?></p>

            <div class="giftcity-wc-sync-actions">
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <?php wp_nonce_field('giftcity_wc_manual_sync'); ?>
                    <input type="hidden" name="action" value="giftcity_wc_manual_sync" />
                    <?php submit_button(__('سنک همه محصولات', 'giftcity-wc'), 'primary', 'submit', false); ?>
                </form>
            </div>

            <div class="giftcity-note">
                <h3><?php esc_html_e('تنظیم سود و قیمت در ووکامرس', 'giftcity-wc'); ?></h3>
                <p><?php esc_html_e('پس از سنک، هر محصول به صورت ساده و مجازی ساخته می‌شود. می‌توانید از بخش «داده‌های محصول» مقدار سود اختصاصی هر محصول را وارد کنید؛ در صورت عدم تنظیم از مقدار پیش‌فرض استفاده می‌شود.', 'giftcity-wc'); ?></p>
            </div>

            <hr />

            <h2><?php esc_html_e('افزودن تکی از لیست محصولات وبسرویس', 'giftcity-wc'); ?></h2>
            <p><?php esc_html_e('از جدول زیر محصول مورد نظر را انتخاب کنید، دسته‌بندی ووکامرس را تعیین کنید و در صورت نیاز یک پراپرتی مشخص را وارد نمایید.', 'giftcity-wc'); ?></p>

            <?php if (empty($remoteProducts)): ?>
                <div class="notice notice-warning">
                    <p><?php esc_html_e('برای مشاهده لیست محصولات، ابتدا کلید API معتبر ثبت کنید و اتصال خود را بررسی نمایید.', 'giftcity-wc'); ?></p>
                </div>
            <?php else: ?>
                <div class="giftcity-remote-products">
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('نام محصول', 'giftcity-wc'); ?></th>
                                <th><?php esc_html_e('دسته‌بندی وبسرویس', 'giftcity-wc'); ?></th>
                                <th><?php esc_html_e('مناطق/ریفیل‌ها', 'giftcity-wc'); ?></th>
                                <th><?php esc_html_e('افزودن به ووکامرس', 'giftcity-wc'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($remoteProducts as $product): ?>
                                <tr>
                                    <td>
                                        <strong><?php echo esc_html($product['name'] ?? __('بدون عنوان', 'giftcity-wc')); ?></strong>
                                        <div class="description">ID: <?php echo esc_html($product['id']); ?></div>
                                    </td>
                                    <td><?php echo esc_html($product['category'] ?? ''); ?></td>
                                    <td>
                                        <?php
                                        $regions = isset($product['regions']) && is_array($product['regions'])
                                            ? wp_list_pluck($product['regions'], 'name')
                                            : [];
                                        echo esc_html(implode('، ', array_filter($regions)));
                                        ?>
                                    </td>
                                    <td>
                                        <?php if (empty($wooCategories)): ?>
                                            <p class="description"><?php esc_html_e('ابتدا یک دسته‌بندی محصول در ووکامرس بسازید.', 'giftcity-wc'); ?></p>
                                        <?php else: ?>
                                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="giftcity-inline-form">
                                                <?php wp_nonce_field('giftcity_wc_import_single'); ?>
                                                <input type="hidden" name="action" value="giftcity_wc_import_single" />
                                                <input type="hidden" name="giftcity_product_id" value="<?php echo esc_attr($product['id']); ?>" />
                                                <label>
                                                    <?php esc_html_e('دسته‌بندی ووکامرس', 'giftcity-wc'); ?>
                                                    <select name="giftcity_wc_term_id" required>
                                                        <option value=""><?php esc_html_e('انتخاب کنید', 'giftcity-wc'); ?></option>
                                                        <?php foreach ($wooCategories as $category): ?>
                                                            <option value="<?php echo esc_attr($category['id']); ?>">
                                                                <?php echo esc_html($category['name']); ?>
                                                            </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </label>
                                                <label>
                                                    <?php esc_html_e('شناسه پراپرتی (اختیاری)', 'giftcity-wc'); ?>
                                                    <input type="number" name="giftcity_property_id" min="1" />
                                                </label>
                                                <?php submit_button(__('افزودن', 'giftcity-wc'), 'secondary', 'submit', false); ?>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>

            <div class="giftcity-note">
                <p><?php esc_html_e('در صورت وارد کردن شناسه پراپرتی، فقط همان پکیج ساخته می‌شود؛ در غیر این صورت تمام پکیج‌های آن محصول به ووکامرس اضافه خواهند شد.', 'giftcity-wc'); ?></p>
            </div>
        </div>
        <?php
    }

    private function getRemoteProducts(): array
    {
        try {
            $result = $this->client->getProducts(1, 50);
            return $result['items'] ?? [];
        } catch (\Throwable $e) {
            return [];
        }
    }

    private function getWooProductCategories(): array
    {
        $terms = get_terms([
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
        ]);

        if (is_wp_error($terms) || empty($terms)) {
            return [];
        }

        return array_map(static function ($term) {
            return [
                'id'   => (int) $term->term_id,
                'name' => $term->name,
            ];
        }, $terms);
    }

    private function maybeGetWalletBalance(): ?array
    {
        try {
            return $this->client->getWalletBalance();
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function getSyncedProductsCount(): int
    {
        $query = new \WP_Query([
            'post_type'      => 'product',
            'posts_per_page' => 1,
            'meta_query'     => [
                [
                    'key'     => '_giftcity_product_id',
                    'compare' => 'EXISTS',
                ],
            ],
        ]);

        $count = (int) ($query->found_posts ?? 0);
        wp_reset_postdata();

        return $count;
    }

    private function translateInterval(string $interval): string
    {
        $map = [
            'giftcity_fifteen_minutes' => __('هر ۱۵ دقیقه', 'giftcity-wc'),
            'hourly' => __('هر ساعت', 'giftcity-wc'),
            'twicedaily' => __('دو بار در روز', 'giftcity-wc'),
            'daily' => __('روزانه', 'giftcity-wc'),
        ];

        return $map[$interval] ?? $interval;
    }
}
